package ags;
import ags.communication.TransferHost;
import gnu.io.CommDriver;
import gnu.io.CommPort;
import gnu.io.CommPortIdentifier;
import gnu.io.RXTXCommDriver;
import gnu.io.SerialPort;
import gnu.io.UnsupportedCommOperationException;
import java.io.IOException;
import java.util.List;
import ags.disk.Disk;
import ags.game.Game;
import ags.game.Games;
import ags.disk.PlainDisk;
import ags.ui.gameSelector.GameSelectorApplication;
import ags.ui.IVirtualScreen;
import ags.ui.TextScreen40;
/**
 * This is the main program for the apple game server.
 * Run this from the command line after putting the apple in input mode.
 * (type "in#2" and press return from the basic prompt and then execute this program)
 * @author blurry
 */
public class Main {
    
    /**
     * List of single-file games
     */
    static List<Game> games;
    /**
     * port being used
     */
    static CommPort port;
    /**
     * Host utility used to communicate with apple
     */
    static TransferHost host;
    /**
     * Constructor (Not used)
     */
    public Main() {}

    /**
     * Initalize com port.  Makes sure it is active.
     * @param portName name of port (e.g. COM2)
     * @throws IOException if port cannot be opened
     */
    private static void initPort(String portName) throws IOException {
        CommDriver driver = new RXTXCommDriver();
        driver.initialize();
        port = driver.getCommPort(portName, CommPortIdentifier.PORT_SERIAL);
        if (port != null) {
            System.out.println("Opened port: "+port.getName());
        } else
            throw new IOException("Port is not available!");
    }
    
    /**
     * This is the main body of the program
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        games = Games.readFromFile();
        System.out.println("Got "+games.size()+" games!");
        String portName = "COM2";
        if (args.length > 0) portName=args[0];
        
        try {
            initPort(portName);
            host = new TransferHost(port);
            host.init();
//            diskHack(host);            
            simpleGameSelector(host);
        } catch (Throwable ex) {
            ex.printStackTrace();
        }
        shutdown();
    }

    /**
     * Experimental code: Serial-based OS hacking
     * @param host Transfer host used to communicate with apple
     * @throws java.io.IOException If there were unrecoverable problems communicating with the apple
     */
    private static void diskHack(TransferHost host) throws IOException {
        Disk d = new PlainDisk("/apple2e/disks/DOS3.3/DOUBLE.TAKE.1.DSK");
        tryDos33Slave(d, host);
    }
    
    /**
     * Experimental code: Serial-based OS hacking replacement for Boot 0 and Boot 1 loaders
     * @param d Disk image to host
     * @param host Transfer host used to communicate with apple
     * @throws java.io.IOException If there were unrecoverable problems communicating with the apple
     */
    private static void tryDos33Slave(Disk d, TransferHost host) throws IOException {
        // First check a few basic things that need to be true
        byte[] sector0 = d.getSector(0,0);
        boolean isDos33 = true;
        // Always one!
        isDos33 &= sector0[0] == 0x01;
        // The command that calls the disk ][ firmware sector read subroutine
        isDos33 &= sector0[0x0036] == 0x6c;
        isDos33 &= sector0[0x0037] == 0x3e;
        isDos33 &= sector0[0x0038] == 0x00;
        
        if (!isDos33) {
            System.out.println("This is not a dos 3.3 disk!");
            return;
        }
        
        // If we got this far, it's time to figure out what boot1 should have loaded
        int numSectors = sector0[255];
        int startAddress = (sector0[254]+numSectors) * 256;
        int[] sectorMap = new int[numSectors+1];
        int currentAddress = startAddress;
        for (int i=0; i <= numSectors; i++) {
            int sectorNum = sector0[0x004d + numSectors - i];
            System.out.println("Sending track 0, sector "+sectorNum+" to address $"+Integer.toHexString(currentAddress));
            host.sendRawData(d.getSector(0, sectorNum), currentAddress, 0, 256);
            currentAddress -= 256;
        }
        System.out.println("Cold start dos (or die trying anyway)");
//        host.jmp(currentAddress + 512);
    }
    
    /**
     * Presents the user with a menu to select a game and then starts the game
     * @param host Transfer host used to communicate with apple
     * @throws java.io.IOException If there were unrecoverable problems communicating with the apple
     */
    private static void simpleGameSelector(TransferHost host) throws IOException {
        IVirtualScreen screen = new TextScreen40();
        GameSelectorApplication app = new GameSelectorApplication(screen);
        app.setGames(games);
        app.mainLoop(host);
    }
    
    /**
     * Close the port and whatever else needs to happen before exiting
     */
    public static void shutdown() {
        ((SerialPort) port).setDTR(false);
        if (port != null) port.close();
    }
}